package gov.va.genisis2.dao.impl;

import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.EntityTransaction;
import javax.persistence.NoResultException;
import javax.persistence.criteria.CriteriaBuilder;
import javax.persistence.criteria.CriteriaQuery;
import javax.persistence.criteria.Predicate;
import javax.persistence.criteria.Root;

import org.apache.commons.lang3.StringUtils;
import org.hibernate.query.Query;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.orm.hibernate5.HibernateTemplate;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import gov.va.genisis2.common.enums.CommonEnum;
import gov.va.genisis2.dao.IRequestDao;
import gov.va.genisis2.model.Request;
import gov.va.genisis2.model.RequestHistory;
import gov.va.genisis2.model.RequestType;
import gov.va.genisis2.model.Source;

/**
 * The Class RequestDao.
 *
 * The Request data access object (DAO) is an object that provides an abstract
 * interface to some type of database or other persistence mechanism. 
 * By mapping application calls to the persistence layer, Request
 * DAO provide some specific data operations without exposing details of the database.
 */
@Repository
@Transactional(value = "transactionManager")
public class RequestDao implements IRequestDao {

	/** The logger. */
	private final org.slf4j.Logger logger = LoggerFactory.getLogger(RequestDao.class);

	/** The hibernate template. */
	@Autowired
	@Qualifier("hibernateTemplate")
	private HibernateTemplate hibernateTemplate;

	/**
	 * Gets the hibernate template.
	 *
	 * @return the hibernate template
	 */
	public HibernateTemplate getHibernateTemplate() {
		return hibernateTemplate;
	}

	/**
	 * Sets the hibernate template.
	 *
	 * @param hibernateTemplate The hibernateTemplate.
	 */
	@Autowired
	public void setHibernateTemplate(HibernateTemplate hibernateTemplate) {
		this.hibernateTemplate = hibernateTemplate;
	}

	/**
	 * This method is used to create request.
	 * @param entity The entity.
	 * @return int This returns created request id.
	 */
	@Override
	public int createRequest(final Request entity) {
		int id = 0;

		if (entity ==null){
			return 0;
		}
		
		entity.setId(this.maxRowValue());
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		EntityTransaction transaction = null;
		try {
			transaction = entityManager.getTransaction();
			transaction.begin();
			entityManager.persist(entity);
			transaction.commit();
			logger.info("RequestDao:  Request Created.");
			id = entity.getId();
		} catch (RuntimeException e) {
			logger.error("RequestDao:  Request Creation failed. Transaction Rolled back.", e);
			if (transaction != null)
				transaction.rollback();
		} finally {
			entityManager.close();
		}
		return id;
	}

	/**
	 * This method is used to get request by UID.
	 * @param uid The uid.
	 * @return this returns list of requestes.
	 */
	@Override
	public List<Request> getRequestsByUID(String uid) {
	
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
		Root<Request> root = query.from(Request.class);
		
		Predicate condition = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_CREATED_BY.getText()), uid);
		Predicate conditions = criteriaBuilder.or(condition);
		query.where(conditions);
		return entityManager.createQuery(query).getResultList();
	}

	/**
	 * This method is used to getRequestsDataManagers.
	 * @param uid The uid.
	 * @return this returns list of requests data managers.
	 */
	@Override
	public List<Request> getRequestsDataManagers(String uid) {
		
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
		Root<Request> root = query.from(Request.class);

		Predicate condition1 = criteriaBuilder.notEqual(root.get(CommonEnum.REQUEST_CREATED_BY.getText()), uid);
		Predicate condition2 = criteriaBuilder.notEqual(root.get(CommonEnum.STATUS_DESC.getText()), CommonEnum.REQUEST_DRAFT.getText());

		Predicate condition3 = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_CREATED_BY.getText()), uid);
		Predicate condition4 = criteriaBuilder.and(condition1, condition2);
		Predicate conditions = criteriaBuilder.or(condition4, condition3);

		query.where(conditions);
		return entityManager.createQuery(query).getResultList();
	}
	
	/**
	 * This method is used to getRequestsByDataManager.
	 * @param uid The uid.
	 * @return this returns list of requests.
	 */
	@Override
	public List<Request> getRequestsByDataManager(String uid) {
		
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
		Root<Request> root = query.from(Request.class);

		Predicate condition = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_CREATED_BY.getText()), uid);
		
		query.where(condition);
		return entityManager.createQuery(query).getResultList();
	}
	
	/**
	 * This method is used to getRequestsDataSourceManager.
	 * @param uid The uid.
	 * @return this returns list of requests data source manager.
	 */
	@Override
	public List<Request> getRequestsDataSourceManagers(String uid) {
	
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
		Root<Request> root = query.from(Request.class);

		Predicate condition1 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), CommonEnum.REQUEST_SENT.getText());
		Predicate condition2 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), CommonEnum.REQUEST_ACCEPTED.getText());
		Predicate condition3 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), CommonEnum.REQUEST_NOT_ACCEPTED.getText());
		
		Predicate conditions = criteriaBuilder.or(condition1, condition2, condition3);

		query.where(conditions);
		
		return entityManager.createQuery(query).getResultList();
	}

	/**
	 * This method is used to get request by id.
	 * @param id The id.
	 * @return Request This returns request.
	 */
	@Override
	public Request getRequestsById(int id) {
		return hibernateTemplate.get(Request.class, id);
	}

	/**
	 * This method is used to get all requests.
	 * @return this returns list of request.
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<Request> getAllRequests() {
		return (List<Request>) hibernateTemplate.find("from Request");
	}

	/**
	 * This method is used to getAllRequestsByStudyApproval.
	 * @param studyApprovalId The studyApprovalId.
	 * @return this returns list of requests.
	 */
	@Override
	public List<Request> getAllRequestsByStudyApproval(int studyApprovalId) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
		Root<Request> root = query.from(Request.class);
		query.where(criteriaBuilder.equal(root.get("studyApproval"), studyApprovalId));
		query.select(root);
		return entityManager.createQuery(query).getResultList();
	}

	/**
	 * This method is used to getAllRequestsByCommentHistory.
	 * @param commentHistoryId The commentHistoryId.
	 * @return this returns list of requests.
	 */
	@Override
	public List<Request> getAllRequestsByCommentHistory(int commentHistoryId) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
		Root<Request> root = query.from(Request.class);
		query.where(criteriaBuilder.equal(root.get("commentHistory"), commentHistoryId));
		query.select(root);
		return entityManager.createQuery(query).getResultList();
	}

	/**
	 * This method is used to getAllRequestsByStaus.
	 * @param status The status.
	 * @return this returns list of requets.
	 */
	@Override
	public List<Request> getAllRequestsByStaus(String status) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery<Request> query = criteriaBuilder.createQuery(Request.class);
		Root<Request> root = query.from(Request.class);
		query.where(criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()), status));
		query.select(root);
		return entityManager.createQuery(query).getResultList();
	}

	/**
	 * This method is used to submit Or Modify entity.
	 * @param entity The entity.
	 * @return int This returns entity's id.
	 */
	@Override
	public int submitOrModify(Request entity) {
		hibernateTemplate.saveOrUpdate(entity);
		return entity.getId();
	}

	/**
	 * This method is used to getRequestType.
	 * @param id The id.
	 * @return RequestType This returns RequestType.
	 */
	@Override
	public RequestType getRequestType(int id) {
		return hibernateTemplate.get(RequestType.class, id);
	}

	/**
	 * This method is used to getSource.
	 * @param id The id.
	 * @return Source This returns Source.
	 */
	@Override
	public Source getSource(int id) {
		return hibernateTemplate.get(Source.class, id);
	}

	/**
	 * Max row value.
	 *
	 * @return int This returns entity's id.
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private int maxRowValue() {
		int maxrowVal = 1;
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery criteriaQuery = criteriaBuilder.createQuery(Request.class);
		Root root = criteriaQuery.from(Request.class);
		criteriaQuery.select(criteriaBuilder.max(root.get("id")));
		Query query = (Query) entityManager.createQuery(criteriaQuery);
		Integer result = (Integer) query.getSingleResult();
		if (result == null)
			return maxrowVal;
		else
			return result.intValue() + 1;
	}

	/**
	 * This method is used to submitOrModify RequestHistory.
	 * @param entity The entity.
	 * @return int This returns RequestHistory id.
	 * 
	 */
	@Override
	public int submitOrModify(RequestHistory entity) {
		int historyLineNo;
		int maxRowHistory;

		if (StringUtils.equals(entity.getStatusDescription(), CommonEnum.REQUEST_DRAFT.getText())) {
			historyLineNo = this.historyLineForRequestHistory(entity.getRequest().getId());
			if (historyLineNo > 0) {
				entity.setHistLineNo(historyLineNo);
			} else {
				maxRowHistory = maxRowValueHistory();
				entity.setHistLineNo(maxRowHistory);
			}

		} else {
			maxRowHistory = maxRowValueHistory();
			entity.setHistLineNo(maxRowHistory);
		}

		hibernateTemplate.saveOrUpdate(entity);
		return entity.getHistLineNo();
	
	}

	/**
	 * Max row value history.
	 *
	 * @return int This returns id.
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private int maxRowValueHistory() {
		int maxRowVal = 1;
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery criteriaQuery = criteriaBuilder.createQuery(RequestHistory.class);
		Root root = criteriaQuery.from(RequestHistory.class);
		criteriaQuery.select(criteriaBuilder.max(root.get("histLineNo")));
		Query query = (Query) entityManager.createQuery(criteriaQuery);
		Integer result = (Integer) query.getSingleResult();

		if (result != null) {
			maxRowVal = result.intValue() + 1;
		}
		
		return maxRowVal;
	}

	/**
	 * This method is used to historyLineForRequestHistory.
	 *
	 * @param requestId The requestid.
	 * @return int This returns requestId.
	 */
	@SuppressWarnings({ "unchecked", "rawtypes" })
	private int historyLineForRequestHistory(int requestId) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery criteriaQuery = criteriaBuilder.createQuery(RequestHistory.class);
		Root root = criteriaQuery.from(RequestHistory.class);

		Predicate condition1 = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_ENTITY.getText()).get("id"), requestId);
		Predicate condition2 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()),CommonEnum.REQUEST_DRAFT.getText());

		Predicate conditions = criteriaBuilder.and(condition1, condition2);

		criteriaQuery.where(conditions);

		RequestHistory result = null;

		Query query = (Query) entityManager.createQuery(criteriaQuery);
		try {
			result = (RequestHistory) query.getSingleResult();
		} catch (NoResultException e) {
			logger.error("Request Id : " + requestId
					+ "  Draft action  in Request History has issues. Could me mutiple instances", e);
		}

		int histLineNo = 0;

		if (result != null) {
			histLineNo = result.getHistLineNo();
		}
		String logInfo = "Request Id : " + requestId + "  Draft present in RequestHistory. HistLineNo: " + histLineNo;
		logger.info(logInfo);
		return histLineNo;
	}

	/**
	 * This method is used to getRequestHistoryById.
	 * @param requestId The requestId.
	 * @return this returns list of requests.
	 */
	@Override
	public List<RequestHistory> getRequestHistoryById(int requestId) {
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder builder = entityManager.getCriteriaBuilder();
		CriteriaQuery<RequestHistory> criteria = builder.createQuery(RequestHistory.class);
		Root<RequestHistory> org = criteria.from(RequestHistory.class);
		criteria.select(org);
		criteria.where(builder.and(builder.equal(org.get(CommonEnum.REQUEST_ENTITY.getText()).get("id"), requestId)));
		criteria.orderBy(builder.desc(org.get("createdOn")));
		return entityManager.createQuery(criteria).getResultList();
	}

	/**
	 * This method is used to getDataSources.
	 * @return this returns list of Source. 
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<Source> getDataSources() {
		return (List<Source>) hibernateTemplate.find("from Source");
	}

	/**
	 * This method is used to getRequestTypes.
	 * @return this returns list of RequestType.
	 */
	@SuppressWarnings("unchecked")
	@Override
	public List<RequestType> getRequestTypes() {
		return (List<RequestType>) hibernateTemplate.find("from RequestType");
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	@Override
	public List<RequestHistory> getRequestHistoryByRequestIdAndStatus(
			int requestId, String status) {
		List<RequestHistory> requestHistory = null;
		
		
		EntityManager entityManager = hibernateTemplate.getSessionFactory().createEntityManager();
		CriteriaBuilder criteriaBuilder = entityManager.getCriteriaBuilder();
		CriteriaQuery criteriaQuery = criteriaBuilder.createQuery(RequestHistory.class);
		Root root = criteriaQuery.from(RequestHistory.class);

		Predicate condition1 = criteriaBuilder.equal(root.get(CommonEnum.REQUEST_ENTITY.getText()).get("id"), requestId);
		Predicate condition2 = criteriaBuilder.equal(root.get(CommonEnum.STATUS_DESC.getText()),status);

		Predicate conditions = criteriaBuilder.and(condition1, condition2);

		criteriaQuery.where(conditions);

		try {
			requestHistory = entityManager.createQuery(criteriaQuery).getResultList();
		} catch (NoResultException ex) {
			logger.error(
					"Error occured while getting Request History for Request Id : "
							+ requestId + " Status:" + status, ex);
		}
		
		return requestHistory;
	}
}
